## Function inputs: q1, ..., q4 = answers to Day 2 elicitation questions
##					expert = character string which is the expert's initials
## Function returns: a vector containing summaries of the prior distributions (incorporating the MYCYC data) of lambda_E, lambda_C, pE, pC and theta.

## 	x[1] = E(pC|zr), x[2] = mode(pC|zr), x[3] = SD(pC|zr), (x[4], x[5]) = 90% credibility interval for pC, x[6] = ESS of prior for log[pC/(1-pC)] incorporating mycyc data
##	x[7] = E(pE|zr), x[8] = mode(pE|zr), x[9] = SD(pE|zr), (x[10], x[11]) = 90% credibility interval for pE incorporating mycyc data
##	x[12] = E(theta|zr), x[13] = mode(theta|zr), x[14] = SD(theta|zr), (x[15], x[16]) = 90% credibility interval for theta, x[17] = ESS of prior for theta incorporating mycyc data
## x[20] = E(lambdaC), x[21] = var(lambdaC), x[22] = E(lambdaE), x[23] = var(lambdaC), x[24] = normalising constant needed for joint prior density of (pE, pC)|zr (zr = mycyc data)

## Note that for considerations of speed, limits of the 90% credibility intervals are set to 0 (and I don't compute x[18] = P(pE > pC), x[19] = P(pE > pC - c2))
## but the code can be easily modified to do so.

priorcall <- function(q1, q2, q3, q4, expert){

	## q1 is the answer to elicitation question i) P{pCR > pC}
	## q2 is the answer to elicitation question ii)  P{pCR < pC - c2}
	## q3 is the answer to elicitation question iii)  P{pER > pE}
	## q4 is the answer to elicitation question iv) P{pER < pE - c2}
	
	c2 = as.double(0.10) 		## trial non-inferiority margin	
	
	## q3 should be less than 1- q4 		
	if( q3 > (1-q4)){
		stop("There is an inconsistency between the answers to elicitation questions Q3 and Q4: Q3 should be less than 1-Q4 \n")
	}
	## q1 should be less than 1- q2 		
	if( q1 > (1-q2)){
		stop("There is an inconsistency between the answers to elicitation questions Q1 and Q2: Q1 should be less than 1-Q2 \n")
	}
	
	## mycyc[1] = Number of successes on CYC in MYCYC database
	## mycyc[2] = Number of failures on CYC in MYCYC dataset
	## mycyc[3] = Number of successes on MMF in MYCYC database
	## mycyc[4] = Number of failures on MMF in MYCYC dataset
	mycyc = scan("MYCYCdata.txt", nmax=4)

  	## Read in parameters of the Beta distribution for pC and Normal distribution for theta 
  	## derived from experts' answers to Day 1 elicitation questions 1-4. 
  	bparam = scan("Day1Prior.txt", nmax=2)
  	thetaqs = scan("Day1Prior.txt", nmax=2, skip = 2)

  	## Call Day 1 code to estimate Day 1 marginal prior for pE
  	x1 = vector(mode="numeric", length = 20)
  	priorParam = vector(mode="numeric", length=14)
  	
  	priorParam[1] = bparam[1]
  	priorParam[2] = bparam[2]
  	priorParam[9] = thetaqs[1]
  	priorParam[10] = thetaqs[2]
  	d1pEprior = Plot_nomycyc(0, 0, 0, 0, x1, priorParam, as.character("pE"), as.logical("T"))
  	
	## determining prior distribution of lambda from answers to elicitation questions
	
  	lCparam = D2prior_lambdaC(q1, 1 - q2, c2, bparam[1], bparam[2])
  	lEparam = D2prior_lambdaE(q3, 1 - q4, c2, d1pEprior)
	
	normEC = D2prior_pcpe(bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc)
	pri_e = D2prior_pe(bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, normEC)
	pri_c = D2prior_pc(bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, normEC)
	pri_t = D2prior_theta(thetaqs[1], thetaqs[2], mycyc, bparam[1], bparam[2], lCparam, lEparam, normEC)

	## Calculating summaries of prior distributions
	x = vector(mode = "numeric", length = 24)

	## Summaries of pC prior distribution (incorporating MYCYC data)
	x[1] = pri_c$expect
	x[2] = pri_c$mode1
	x[3] = pri_c$sdev
	x[4] = pri_c$ci_low
	x[5] = pri_c$ci_upp
	x[6] = D2ess_pc(bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, x[1], normEC)
		
	## Summaries of pE prior distribution (incorporating MYCYC data)
	x[7] = pri_e$expect
	x[8] = pri_e$mode1
	x[9] = pri_e$sdev
	x[10] = pri_e$ci_low
	x[11] = pri_e$ci_upp 
	
	## Summaries of theta prior distribution (incorporating MYCYC data)

	x[12] = pri_t$expect
	x[13] = pri_t$mode1
	x[14] = pri_t$sdev
	x[15] = pri_t$ci_low
	x[16] = pri_t$ci_upp
	x[17] = D2ess_theta(1, thetaqs[1], thetaqs[2], bparam[1], bparam[2], lCparam, lEparam, mycyc, normEC, x[14]^2)	

	x[24] = normEC

	## Summaries of lambdaC and lambdaE prior distributions (means and variances)
	x[20] = lCparam[1]
	x[21] = lCparam[2]
	x[22] = lEparam[1]
	x[23] = lEparam[2]
	
	## Write answers to Day 2 elicitation questions to file 
	outputfile <- paste(expert, "-D2answer.txt", sep="")
	cat(expert, " answers to Day 2 prior elicitation questions were: \n", file = outputfile, append=FALSE)
	cat(q1, file = outputfile, sep="\n", append=TRUE)
	cat(q2, file = outputfile, sep="\n", append=TRUE)
	cat(q3, file = outputfile, sep="\n", append=TRUE)
	cat(q4, file = outputfile, sep="\n", append=TRUE)

	return(x)
}

## Function inputs:	n_mmf, mmf_succ = number randomised to MMF and number of successful outcomes on MMF
##					n_cyc, cyc_succ = number randomised to CYC and number of successful outcomes on CYC
##					priorParm = vector containing output from priorcall
##					posterior40 = indicator of whether posterior distributions are calculated assuming data are available on a total of 40 patients.
## Function returns: 	vector containing summaries of the posterior distributins of pC, pE and theta
##	x[1] = E(pC|data, zr), x[2] = mode(pC|data, zr), x[3] = SD(pC|data, zr), (x[4], x[5]) = 90% posterior credibility interval for pC, x[6] = normalising constant of joint posterior density of (pC, pE)|MYPAN data and mycyc data.
## 	x[7] = E(pE|data, zr), x[8] = mode(pE|data, zr), x[9] = SD(pE|data, zr), (x[10], x[11]) = 90% posterior credibility interval for pE
##	x[12] = E(theta|data, zr), x[13] = mode(theta|data, zr), x[14] = SD(theta|data, zr), (x[15], x[16]) = 90% posterior credibility interval for theta, 
##	x[19] = normalising constant of joint posterior density of (pC, theta)|MYPAN data and mycyc data.

## Note that for considerations of speed, limits of the 90% credibility intervals are set to 0 (and I don't compute x[17] = P(pE > pC|zr, data), 
## x[18] = P(pE > pC - 0.1|zr, data)) but the code can be easily modified to do so.

postcall <- function(n_mmf, mmf_succ, n_cyc, cyc_succ, priorParm, posterior40){
	 		
	c2 = as.double(0.10) 		## Non-inferiority margin for the trial	
		
	## Catching possible input errors for mmf_succ, n_cyc and cyc_succ
	if(mmf_succ > n_mmf){
		stop("Data on MMF: number of successes exceeds number randomised to MMF")
	}
	
	if(posterior40 & !identical(as.integer(n_mmf + n_cyc), as.integer(40))){
		stop("Total number randomized to MMF and CYC does not sum to 40")
	}
	if(!posterior40 & !identical(as.integer(n_mmf + n_cyc), as.integer(20))){
		stop("Total number randomized to MMF and CYC does not sum to 20")
	}
	
  	## Read in parameters of the Beta distribution for pC and Normal distribution for theta 
  	## derived from experts' answers to Day 1 elicitation questions 1-4. 
  	bparam = scan("Day1Prior.txt", nmax=2)
  	thetaqs = scan("Day1Prior.txt", nmax=2, skip = 2)
  	
	## mycyc[1] = Number of successes on CYC in MYCYC database
	## mycyc[2] = Number of failures on CYC in MYCYC dataset
	## mycyc[3] = Number of successes on MMF in MYCYC database
	## mycyc[4] = Number of failures on MMF in MYCYC dataset
	mycyc = scan("MYCYCdata.txt", nmax=4)
	
	lCparam = c(priorParm[20], priorParm[21])
	lEparam = c(priorParm[22], priorParm[23])
	
	postpe = D2post_pe(mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc-cyc_succ, bparam[1], bparam[2], lCparam, lEparam, thetaqs[1], thetaqs[2], mycyc, priorParm[24])
	postpc = D2post_pc(mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc-cyc_succ, bparam[1], bparam[2], lCparam, lEparam, thetaqs[1], thetaqs[2], mycyc, priorParm[24])
	post_t = D2post_theta(mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc-cyc_succ, bparam[1], bparam[2],lCparam, lEparam, thetaqs[1], thetaqs[2], mycyc, priorParm[24])
	
	x = vector(mode="numeric", length = 19)
	## Summaries of pC posterior distribution
	x[1] = postpc$expect
	x[2] = postpc$mode1
	x[3] = postpc$sdev
	x[4] = postpc$ci_low
	x[5] = postpc$ci_upp
	x[6] = postpc$postNormEC
	
	## Summaries of pE posterior distribution
	x[7] = postpe$expect
	x[8] = postpe$mode1
	x[9] = postpe$sdev
	x[10] = postpe$ci_low
	x[11] = postpe$ci_upp 
	
	## Summaries of theta posterior distribution
	x[12] = post_t$expect
	x[13] = post_t$mode1
	x[14] = post_t$sdev
	x[15] = post_t$ci_low
	x[16] = post_t$ci_upp
	x[19] = post_t$normTC	

	return(x)
}

## Functions to evaluate the prior and posterior densities of pC, pE and theta
## Function inputs:	n_mmf, mmf_succ = number of patients randomised to MMF and number of successful outcomes observed
## 					n_cyc, cyc_succ = number randomised to CYC and number of successful outcomes observed
## 					postParm = vector of outputs returned by postcall function
## 					priorParm = vector of outputs returned by priorcall function
## 					parmInd = character string indicating parameter we wish to summarise
## 					postind = whether we wish to plot the prior or posterior density of the stated parameter.
## Function returns: a data frame containing a vector of values of the parameter and the prior or posterior density evaluated at those values. 

distPlot <- function(n_mmf, mmf_succ, n_cyc, cyc_succ, postParm, priorParm, parmInd, postind){

	r = as.integer(16)
  	mesh = as.integer(6*r -1)
  	mesh1 = as.integer(2*mesh-1)

	## mycyc[1] = Number of successes on CYC in MYCYC database
	## mycyc[2] = Number of failures on CYC in MYCYC dataset
	mycyc = scan("MYCYCdata.txt", nmax=4)
	
	## Read in parameters of the Beta distribution for pC from file
  	## Beta distribution based on Day 1 elicitation exercise
  	bparam = scan("Day1Prior.txt", nmax=2)
  	thetaqs = scan("Day1Prior.txt", nmax=2, skip = 2)
  	
  	lCparam = c(priorParm[20], priorParm[21])
	lEparam = c(priorParm[22], priorParm[23])
	
	if(postind){
		
		## Wish to generate prior density for the stated parameter
		if(parmInd == "pC"){
			## Evaluate the prior density of pC (incorporating historical data)
      		gridc = seq(0.01, 0.99, by=0.01)
			gridc = append(c(0.00001, 0.001), gridc)
			gridc = append(gridc, c(0.999, 0.99999))	
			dens = D2pc_dens(gridc, priorParm[24], postParm[6], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, bparam[1], bparam[2], 0,0,0,0, as.logical("F"))
			return(data.frame(gridc, dens))
		}else if(parmInd=="pE"){
			## Evaluate the prior density of pE (incorporating historical data)
      		gride = seq(0.01, 0.99, by=0.01)
			gride = append(c(0.00001, 0.001), gride)
			gride = append(gride, c(0.999, 0.99999))
			## Call function to evaluate pE prior density at grid of values
			dens = D2pe_dens(gride, priorParm[24], postParm[6], bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, 0, 0, 0, 0, as.logical("F"))
      		return(data.frame(gride, dens))		
		}else if(parmInd == "lambdaC"){
			## Evaluate the prior density of lambdaC
      		gridl = seq(-3, 3, by = 0.05)	
			dens = dnorm(gridl, mean=priorParm[20], sd=sqrt(priorParm[21]), log=FALSE)
			return(data.frame(gridl, dens))	
		}else if(parmInd == "lambdaE"){
			## Evaluate the prior density of lambdaC
      		gridl = seq(-3, 3, by = 0.05)	
			dens = dnorm(gridl, mean=priorParm[22], sd=sqrt(priorParm[23]), log=FALSE)
			return(data.frame(gridl, dens))	
		}else{	
			## Evaluate the prior density of theta (incorporating historical data)
			gridt = seq(-2, 2, by = 0.05)	
			dens = D2theta_dens(gridt, priorParm[24], postParm[19], bparam[1], bparam[2], lCparam, lEparam, mycyc, thetaqs[1], thetaqs[2], 0, 0, 0, 0, as.logical("F"))
			return(data.frame(gridt, dens))	
		}	
	}else{
		## Wish to generate posterior density for the stated parameter
		if(parmInd=="pC"){
			gridc = seq(0.01, 0.99, by=0.01)
			gridc = append(c(0.00001, 0.001), gridc)
			gridc = append(gridc, c(0.999, 0.99999))	
			postd = D2pc_dens(gridc, priorParm[24], postParm[6], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, bparam[1], bparam[2], mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc - cyc_succ, as.logical("T"))
			
			return(data.frame(gridc, postd))
		}else if(parmInd == "pE"){
			gride = seq(0.01, 0.99, by=0.01)
			gride = append(c(0.00001, 0.001), gride)
			gride = append(gride, c(0.999, 0.99999))
			postd = D2pe_dens(gride, priorParm[24], postParm[6], bparam[1], bparam[2], lCparam, lEparam, 1/sqrt(thetaqs[2]), thetaqs[1]/sqrt(thetaqs[2]), mycyc, mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc - cyc_succ, as.logical("T"))
			
			return(data.frame(gride, postd))
		}else{
			gridt = seq(-2, 2, by = 0.05)
			postd = D2theta_dens(gridt, priorParm[24], postParm[19], bparam[1], bparam[2], lCparam, lEparam, mycyc, thetaqs[1], thetaqs[2], mmf_succ, n_mmf - mmf_succ, cyc_succ, n_cyc - cyc_succ, as.logical("T"))
			return(data.frame(gridt, postd))
		}
	}		
}

## Evaluate prior or posterior density of pC (incorporating historical data)
## Function inputs: 	gridc = grid of values of pC
##						priNormEC, postnormEC = 
##						lCparam, lEparam = vectors containing prior mean and variances of lambdaC and lambdaE
##						sigmainv, musigma = 1/SD(theta), E(theta)/SD(theta) (where summaries are with respect to the Day 1 theta prior not incorporating the MYCYC data)
##						mycyc = vector containing summaries of mycyc dataset
##						a,b = parameters of Beta(a,b) Day 1 prior distribution for pC (not incorporating mycyc data)
##						se, fe, sc, fc  = summaries of hypothetical mypan trial dataset
##						postind = indicator of whether want to summarise prior/posterior density 
## Function returns: 	vector of values of prior/posterior density of pC evaluated at gridc (priors and posteriors incorporate mycyc data)
D2pc_dens <- function(gridc, priNormEC, postNormEC, lCparam, lEparam, sigmainv, musigma, mycyc, a, b, se, fe, sc, fc, postInd){
	
	lc = length(gridc)
	gride = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gride = append(c(0.00001, midp1), gride)
    gride = append(gride, c(midp2, 0.99999))
  	le = length(gride)
  	we  = vector(mode="numeric", length=le)
  	we[1] = (gride[3]-gride[1])/6.0
  	we[le] = (gride[le] - gride[le-2])/6.0
  	for(i in seq(2, le-1, by=2)){
    	we[i] = 4*(gride[i+1] - gride[i-1])/6.0
  	}
  	for(i in seq(3, le-2, by=2)){
    	we[i] = (gride[i+2] - gride[i-2])/6.0
  	}
  	
  	gridea = gride
  	gridca = gride
  	intpea = vector(mode="numeric", le)
  	denspc = vector(mode="numeric", le)
	for(j in 1:le){
  		const1 = dnorm(log(gridea*(1-gride[j])/(gride[j]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	int = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea[j] = sum(we*int)
  	}
  	
	if(!postInd){
		## Evaluate prior density of pC incorporating MYCYC data
  		dens = vector(mode="numeric", lc)
  		for(i in 1:lc){
  			const1 = dnorm(log(gridca*(1-gridc[i])/(gridc[i]*(1-gridca))), mean=lCparam[1], sd = sqrt(lCparam[2]))
    		int = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    		intpca = sum(we*int)
  			for(j in 1:le){	
    			const1 = dnorm(log(gride[j]*(1-gridc[i])/(gridc[i]*(1 - gride[j]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    			denspc[j] = priNormEC*const1*intpca*intpea[j]*dbeta(gridc[i], shape1=a, shape2=b)/(gride[j]*(1-gride[j]))
  			}
  			dens[i] = sum(we*denspc)
  		}
		return(dens)
	}else{
		postd = vector(mode="numeric", length = lc)
  		for(i in 1:lc){
  			const1 = dnorm(log(gridca*(1-gridc[i])/(gridc[i]*(1-gridca))), mean=lCparam[1], sd = sqrt(lCparam[2]))
    		int = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    		intpca = sum(we*int)
  			for(j in 1:le){	
    			const1 = dnorm(log(gride[j]*(1-gridc[i])/(gridc[i]*(1 - gride[j]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    			denspc[j] = priNormEC*const1*intpca*intpea[j]*dbeta(gridc[i], shape1=a, shape2=b)/(gride[j]*(1-gride[j]))
    			denspc[j] = denspc[j]*(gride[j]^se)*((1-gride[j])^fe)
  			}
  			postd[i] = sum(we*denspc)*(gridc[i]^sc)*((1-gridc[i])^fc)*postNormEC
  		}
		return(postd)
	}	
}

## Evaluate prior or posterior density of pE at grid of points in gride (incorporating historical data)
D2pe_dens <- function(gride, priNormEC, postNormEC, a, b, lCparam, lEparam, sigmainv, musigma, mycyc, se, fe, sc, fc, postInd){
	
	le = length(gride)
	gridc = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))
  	lc = length(gridc)
  	wc  = vector(mode="numeric", length=lc)
  	wc[1] = (gridc[3]-gridc[1])/6.0
  	wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
  	for(i in seq(2, lc-1, by=2)){
    	wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
  	}
  	for(i in seq(3, lc-2, by=2)){
    	wc[i] = (gridc[i+2] - gridc[i-2])/6.0
  	}
  	
  	gridea = gridc
  	gridca = gridc
  	intpca = vector(mode="numeric", lc)
  	denspc = vector(mode="numeric", lc)
  	for(j in 1:lc){
  		const1 = dnorm(log(gridca*(1-gridc[j])/(gridc[j]*(1-gridca))), mean = lCparam[1], sd = sqrt(lCparam[2]))
    	int = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca[j] = sum(wc*int)
  	}
	if(!postInd){
		## Evaluate prior density of pE incorporating MYCYC data
  		dens = vector(mode="numeric", le)
  		for(i in 1:le){
  			const1 = dnorm(log(gridea*(1-gride[i])/(gride[i]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    		int = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
			intpea = sum(wc*int)
  			for(j in 1:lc){	
    			const1 = dnorm(log(gride[i]*(1-gridc[j])/(gridc[j]*(1 - gride[i]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    			denspc[j] = priNormEC*const1*intpca[j]*intpea*dbeta(gridc[j], shape1=a, shape2=b)/(gride[i]*(1-gride[i]))
  			}
  			dens[i] = sum(wc*denspc)
  		}
  		
		return(dens)
	}else{
		postd = vector(mode="numeric", length = le)
  		for(i in 1:le){
  			const1 = dnorm(log(gridea*(1-gride[i])/(gride[i]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    		int = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
			intpea = sum(wc*int)
  			for(j in 1:lc){	
    			const1 = dnorm(log(gride[i]*(1-gridc[j])/(gridc[j]*(1 - gride[i]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    			denspc[j] = priNormEC*const1*intpca[j]*intpea*dbeta(gridc[j], shape1=a, shape2=b)/(gride[i]*(1-gride[i]))
    			denspc[j] = denspc[j]*(gridc[j]^sc)*((1-gridc[j])^fc)
  			}
  			postd[i] = sum(wc*denspc)*(gride[i]^se)*((1-gride[i])^fe)*postNormEC
  		}
		return(postd)
	}
	
}

## Evaluate posterior density of theta
## Function input: priNormEC = 
##					postNormTC = 
## Function returns: 	vector of values of prior/posterior density of theta evaluated at gridt (priors and posteriors incorporate mycyc data)
D2theta_dens <- function(gridt, priNormEC, postNormTC, a, b, lCparam, lEparam, mycyc, mu, sigma2, se, fe, sc, fc, postInd){
	
	lt = length(gridt)
	gridc = seq(0.001, 0.999, by=0.001)
	midp1 = (0.00001 + 0.001)*0.5
	midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))  
    lc = length(gridc)

    wc  = vector(mode="numeric", length=lc)
    wc[1] = (gridc[3]-gridc[1])/6.0
    wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
    for(i in seq(2, lc-1, by=2)){
		wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
	}
    for(i in seq(3, lc-2, by=2)){
    	wc[i] = (gridc[i+2] - gridc[i-2])/6.0
	} 
	
	grider = gridc
	gridcr = gridc
	denspc = vector(mode="numeric", length=lc)
  	int1 = (grider^(mycyc[3] -1))*((1-grider)^(mycyc[4]-1))
  	int2 = (gridcr^(mycyc[1]-1))*((1-gridcr)^(mycyc[2]-1))
	if(!postInd){
  		dens = vector(mode="numeric", length=lt)
  		for(i in 1:lt){
  		## Integrate over joint density of (theta, pC)|zr	
  			for(j in 1:lc){
  				int3 = int1*dnorm(log(grider*(1-gridc[j])/((1-grider)*gridc[j]*exp(gridt[i]))), mean = lEparam[1], sd = sqrt(lEparam[2]))
  				int4 = int2*dnorm(log(gridcr*(1-gridc[j])/((1-gridcr)*gridc[j])), mean = lCparam[1], sd = sqrt(lCparam[2]))
  				denspc[j] = sum(wc*int3)*sum(wc*int4)*dbeta(gridc[j], shape1 = a, shape2=b)
  			}
  			dens[i] = dnorm(gridt[i], mean=mu, sd = sqrt(sigma2))*priNormEC*sum(wc*denspc)
		}
		return(dens)	
	}else{
		postd = vector(mode="numeric", length=lt)
  		for(i in 1:lt){
  		## Integrate over joint density of (theta, pC)|zr	
  			for(j in 1:lc){
  				phi = gridc[j]*exp(gridt[i])/(gridc[j]*(exp(gridt[i]) -1) +1)
  				int3 = int1*dnorm( log(grider*(1-phi)/(phi*(1-grider))), mean = lEparam[1], sd = sqrt(lEparam[2]))
  				int4 = int2*dnorm(log(gridcr*(1-gridc[j])/((1-gridcr)*gridc[j])), mean = lCparam[1], sd = sqrt(lCparam[2]))
  				denspc[j] = sum(wc*int3)*sum(wc*int4)*dbeta(gridc[j], shape1 = a, shape2=b)*priNormEC/(phi*(1-phi))
  				denspc[j] = denspc[j]*dnorm(log(phi*(1-gridc[j])/(gridc[j]*(1-phi))), mean = mu, sd = sqrt(sigma2))
  				denspc[j] = denspc[j]*(gridc[j]^(se+sc+1))*((1-gridc[j])^(fe+fc+1))/((gridc[j]*(exp(gridt[i]) -1) +1)^(se+fe+2))
  			}
  			postd[i] = sum(wc*denspc)*exp((se+1)*gridt[i])*postNormTC
		}
		return(postd)
	}
	
}